#
# This is a Shiny web application. You can run the application by clicking
# the 'Run App' button above.
#
# Find out more about building applications with Shiny here:
#
#    http://shiny.rstudio.com/
#

library(shiny)
library(ggplot2)
source("helpers.R")


#### Define UI ####
ui <- fluidPage(
  
  tags$head(
    tags$link(rel = "stylesheet", type = "text/css", href = "style.css")
  ),
  
  tags$style(HTML(".js-irs-0 .irs-single, .js-irs-0 .irs-bar-edge, .js-irs-0 .irs-bar {background: #ffc125}")),
  tags$style(HTML(".js-irs-1 .irs-single, .js-irs-1 .irs-bar-edge, .js-irs-1 .irs-bar {background: #ffc125}")),
  tags$style(HTML(".js-irs-2 .irs-single, .js-irs-2 .irs-bar-edge, .js-irs-2 .irs-bar {background: #ffc125}")),
  tags$style(HTML(".js-irs-3 .irs-single, .js-irs-3 .irs-bar-edge, .js-irs-3 .irs-bar {background: #b22222}")),
  tags$style(HTML(".js-irs-4 .irs-single, .js-irs-4 .irs-bar-edge, .js-irs-4 .irs-bar {background: #b22222}")),  
  tags$style(HTML(".js-irs-5 .irs-single, .js-irs-5 .irs-bar-edge, .js-irs-5 .irs-bar {background: #b22222}")),
  
  # Application title
   titlePanel(h1("Robustness and power of the",em(" t"), "test"), windowTitle = "Unequal variance and t-tests"),
   
   #### Input sidebar ####
   sidebarLayout(
      sidebarPanel(
        radioButtons("classic_test", "Choose which type of t-test to use:",
                     list("Classic 2-sample" = TRUE, "Welch's" = FALSE),
                     inline=TRUE),
        h4("Population 1", style = "color:#ffc125"),
        
        sliderInput("mean1", "Mean:",
                    min = 5, max = 15, value = 10, step = 0.5,
                    ticks=FALSE,
                    width="200px"),
        
        sliderInput("sd1", "Standard deviation:",
                    min = 1, max = 5, value = 1.5, step = 0.5,
                    ticks=FALSE,
                    width="200px"),
       
       sliderInput("sample_size_1", "Sample size:",
                   min = 2, max = 100, value = 25,
                   ticks=FALSE,
                   width="200px"),

      h4("Population 2", style = "color: #b22222"),
       
      sliderInput("mean2", "Mean:",
                   min = 5, max = 15, value = 10, step = 0.5,
                   ticks=FALSE,
                   width="200px"),
        
      sliderInput("sd2", "Standard deviation:",
                  min = 1, max = 5, value = 2, step = 0.5,
                  ticks=FALSE,
                  width="200px"),
        
      sliderInput("sample_size_2", "Sample size:",
                  min = 2, max = 100, value = 25,
                  ticks=FALSE,
                  width="200px")
      ),
      
     
      
      ##### Show a plot of the generated distribution ####
      mainPanel(
 
        htmlOutput("testHTML"),
        p("Each time a parameter is changed, new simulations are run automatically. α is set to 5%.", style = "font-size:80%"),
      
        h3(textOutput("summary"),style="color:red"),
        plotOutput("twoDist", height = "200px"),
        br(),
        h4(HTML("Distribution of <em>t</em>'s from simulation results. <em>P</em> < 0.05 in <span style= 'color:darkblue'>dark blue</span>.")),
        plotOutput("distPlot", height = "200px")
      )
   ),
  HTML("<div id = \"credits\">Developed by <a href=\"http://www.zoology.ubc.ca/~whitlock/\">Michael Whitlock</a>  
        at the University of British Columbia. Look 
       <a href=\"http://www.zoology.ubc.ca/~whitlock/kingfisher/KFhomepage.htm \">here</a> for more apps. This web-page and its code is released on a 
       <a href=\"https://wiki.creativecommons.org/wiki/CC0\">Creative Commons Zero</a> 
       agreement, meaning that it is freely available for use, re-use, and modification. 
       We request that you give credit, when possible. This work is in the public domain.</p></div>")
)

##### Define server ####
server <- function(input, output) {
  
  output$testHTML <- renderUI({
    if(input$mean1==input$mean2){
      HTML(paste("<h4>",numreps,"simulations were run. The null hypothesis of equal means is <span style='color:red'>true</span>.</h4>"))
    }
    else {
      HTML(paste("<h4>",numreps,"simulations were run. The null hypothesis of equal means is <span style='color:red'>false</span>.</h4>"))      
    }
  })
  
  tlist <- reactive(
    multiple_tests(numreps,"normal",input$sample_size_1,input$mean1,input$sd1,"normal",input$sample_size_2,input$mean2,input$sd2,var.equal=as.logical(input$classic_test))
    #multiple_tests(numreps,input$type1,input$sample_size_1,mean1,input$sd1,input$type2,input$sample_size_2,mean2,input$sd2,var.equal=TRUE)
    )
  
  
  resultText <- reactive({
    if(input$mean1==input$mean2){
      paste("The Type I error rate was",as.numeric(typeIerror(tlist())),".")
    } else{
      paste("The power was",as.numeric(typeIerror(tlist())),".")
    }
  })
  
  # Return the  text outputs
  output$simInfo <- renderText({
    simInfoText()
  })
  
  output$summary <- renderText({
    resultText()
  })
  
  #Add the plot of the two population distributions
  output$twoDist <- renderPlot({
    
    twoNormalPlotter(input$mean1, input$sd1, input$mean2, input$sd2, xMin=0, xMax=20)
    
  })
  
  # Add the plot of the sampling distribution of t
   output$distPlot <- renderPlot({
      
      tDistPlotter(tlist())
      
   })
}

# Run the application 
shinyApp(ui = ui, server = server)


